package org.msh.tb.bd.tbforms.indicator.tb10;

import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.msh.tb.bd.tbforms.query.TbFormQuery;
import org.msh.tb.bd.tbforms.query.tb10.TBForm10Block1Query;
import org.msh.tb.entities.AdministrativeUnit;
import org.msh.tb.entities.Tbunit;
import org.msh.tb.entities.enums.*;
import org.msh.tb.indicators.core.Indicator2D;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

@Name("TBForm10Block1")
public class TBForm10Block1 extends Indicator2D {

    private TB10Block1Info block1Info;

    private TbFormQuery formQuery;

    @In(create=true) protected Map<String, String> messages;

    private TbFormQuery getFormQuery() {
        if (formQuery == null) {
            formQuery = new TBForm10Block1Query();
        }

        return formQuery;
    }
    
    private TB10Block1Info getBlock1Info() {
        if (block1Info == null) {
            block1Info = new TB10Block1Info();
        }
        
        return block1Info;
    }

	@Override
	protected void createIndicators() {

        Tbunit tbunit = getIndicatorFilters().getTbunitselection().getSelected();
        AdministrativeUnit adminUnit = getIndicatorFilters().getTbunitselection().getLastLevelAdminUnitSelected();

        List<Object[]> result = getFormQuery().queryDB(tbunit,
                adminUnit,
                getWorkspace(),
                getIndicatorFilters().getQuarter(),
                getEntityManager());

        allocateValuesOnFields(result);

        populateInterfaceTableRows();
	}

    /**
     * The logic for allocating the values from the result of the queries on the object that stored this results was encapsulated on this method.
     * @param result the result returned from the query
     */
    public void allocateValuesOnFields(List<Object[]> result){
        for(Object[] o : result){
            PatientType pt1 = (PatientType) o[0];
            PatientType prevpt = (PatientType) o[1];
            CaseDefinition cd = (CaseDefinition) o[2];
            InfectionSite is = (InfectionSite) o[3];
            Integer age = (Integer) o[4];
            Gender g = (Gender) o[5];
            Long qtd = (Long) o[6];

            PatientType pt = (pt1.equals(PatientType.PREVIOUSLY_TREATED) ? prevpt : pt1);

            if (pt != null) {
                int subgroup = -1;

                if (InfectionSite.EXTRAPULMONARY.equals(is)) {
                    subgroup = 2;
                } else if (InfectionSite.PULMONARY.equals(is) && CaseDefinition.BACTERIOLOGICALLY_CONFIRMED.equals(cd)) {
                    subgroup = 0;
                } else if (InfectionSite.PULMONARY.equals(is) && CaseDefinition.CLINICALLY_DIAGNOSED.equals(cd)) {
                    subgroup = 1;
                }

                getBlock1Info().setValue(Block1AgeRange.findAgeRange(age), subgroup, g, pt, qtd);
            }
        }
    }

    /**
     * Write the results allocated on the block1info object to the interface table object
     */
    public void populateInterfaceTableRows(){
        // conditions sent to UI for a detailed case list
        getTable()
                .addCondition("c.classification = " + CaseClassification.TB.ordinal())
                .addCondition("c.diagnosisType = " + DiagnosisType.CONFIRMED.ordinal());

        // Populates the age ranges row
        for(Block1AgeRange key : getBlock1Info().getAgeGroupsRows().keySet()){
            //Pulmonary - Bacteriologically Confirmed
            for(PatientType pt : getPatientTypesReport()){
                addValue("M"+pt.ordinal()+"bc",
                        messages.get("manag.gender.male"),
                        key.toString(),
                        getBlock1Info().getAgeGroupsRows().get(key).getPulmonaryBacteriologicallyConf().getMaleValues().get(pt).floatValue())
                        // conditions sent to UI for a detailed case list
                        .addCondition("c.age >= " + key.iniAge)
                        .addCondition("c.age <= " + key.endAge)
                        .addCondition("p.gender = " + Gender.MALE.ordinal())
                        .addCondition("c.infectionSite = " + InfectionSite.PULMONARY.ordinal())
                        .addCondition("c.caseDefinition = " + CaseDefinition.BACTERIOLOGICALLY_CONFIRMED.ordinal())
                        .addCondition("(c.patientType = " + pt.ordinal() + " or c.previouslyTreatedType = " + pt.ordinal() + ")");

                addValue("F"+pt.ordinal()+"bc",
                        messages.get("manag.gender.female"),
                        key.toString(),
                        getBlock1Info().getAgeGroupsRows().get(key).getPulmonaryBacteriologicallyConf().getFemaleValues().get(pt).floatValue())
                        // conditions sent to UI for a detailed case list
                        .addCondition("c.age >= " + key.iniAge)
                        .addCondition("c.age <= " + key.endAge)
                        .addCondition("p.gender = " + Gender.FEMALE.ordinal())
                        .addCondition("c.infectionSite = " + InfectionSite.PULMONARY.ordinal())
                        .addCondition("c.caseDefinition = " + CaseDefinition.BACTERIOLOGICALLY_CONFIRMED.ordinal())
                        .addCondition("(c.patientType = " + pt.ordinal() + " or c.previouslyTreatedType = " + pt.ordinal() + ")");
            }

            //Pulmonary - Clinically Confirmed
            for(PatientType pt : getPatientTypesReport()){
                addValue("M" + pt.ordinal() + "cc",
                        messages.get("manag.gender.male"),
                        key.toString(),
                        getBlock1Info().getAgeGroupsRows().get(key).getPulmonaryClinicallyConf().getMaleValues().get(pt).floatValue())
                        // conditions sent to UI for a detailed case list
                        .addCondition("c.age >= " + key.iniAge)
                        .addCondition("c.age <= " + key.endAge)
                        .addCondition("p.gender = " + Gender.MALE.ordinal())
                        .addCondition("c.infectionSite = " + InfectionSite.PULMONARY.ordinal())
                        .addCondition("c.caseDefinition = " + CaseDefinition.CLINICALLY_DIAGNOSED.ordinal())
                        .addCondition("(c.patientType = " + pt.ordinal() + " or c.previouslyTreatedType = " + pt.ordinal() + ")");

                addValue("F"+pt.ordinal()+"cc",
                        messages.get("manag.gender.female"),
                        key.toString(),
                        getBlock1Info().getAgeGroupsRows().get(key).getPulmonaryClinicallyConf().getFemaleValues().get(pt).floatValue())
                        // conditions sent to UI for a detailed case list
                        .addCondition("c.age >= " + key.iniAge)
                        .addCondition("c.age <= " + key.endAge)
                        .addCondition("p.gender = " + Gender.FEMALE.ordinal())
                        .addCondition("c.infectionSite = " + InfectionSite.PULMONARY.ordinal())
                        .addCondition("c.caseDefinition = " + CaseDefinition.CLINICALLY_DIAGNOSED.ordinal())
                        .addCondition("(c.patientType = " + pt.ordinal() + " or c.previouslyTreatedType = " + pt.ordinal() + ")");
            }

            //Extrapulmonary
            for(PatientType pt : getPatientTypesReport()){
                addValue("M" + pt.ordinal() + "ex",
                        messages.get("manag.gender.male"), key.toString(),
                        getBlock1Info().getAgeGroupsRows().get(key).getExtrapulmonary().getMaleValues().get(pt).floatValue())
                        // conditions sent to UI for a detailed case list
                        .addCondition("c.age >= " + key.iniAge)
                        .addCondition("c.age <= " + key.endAge)
                        .addCondition("p.gender = " + Gender.MALE.ordinal())
                        .addCondition("c.infectionSite = " + InfectionSite.EXTRAPULMONARY.ordinal())
                        .addCondition("(c.patientType = " + pt.ordinal() + " or c.previouslyTreatedType = " + pt.ordinal() + ")");

                addValue("F"+pt.ordinal()+"ex",
                        messages.get("manag.gender.female"),
                        key.toString(), getBlock1Info().getAgeGroupsRows().get(key).getExtrapulmonary().getFemaleValues().get(pt).floatValue())
                        // conditions sent to UI for a detailed case list
                        .addCondition("c.age >= " + key.iniAge)
                        .addCondition("c.age <= " + key.endAge)
                        .addCondition("p.gender = " + Gender.FEMALE.ordinal())
                        .addCondition("c.infectionSite = " + InfectionSite.EXTRAPULMONARY.ordinal())
                        .addCondition("(c.patientType = " + pt.ordinal() + " or c.previouslyTreatedType = " + pt.ordinal() + ")");
            }

            addValue("MT",
                    messages.get("manag.gender.male"),
                    key.toString(),
                    getBlock1Info().getAgeGroupsRows().get(key).getTotalByGender(Gender.MALE).floatValue())
                    // conditions sent to UI for a detailed case list
                    .addCondition("c.age >= " + key.iniAge)
                    .addCondition("c.age <= " + key.endAge)
                    .addCondition("p.gender = " + Gender.MALE.ordinal());
            addValue("FT",
                    messages.get("manag.gender.female"),
                    key.toString(),
                    getBlock1Info().getAgeGroupsRows().get(key).getTotalByGender(Gender.FEMALE).floatValue())
                    // conditions sent to UI for a detailed case list
                    .addCondition("c.age >= " + key.iniAge)
                    .addCondition("c.age <= " + key.endAge)
                    .addCondition("p.gender = " + Gender.FEMALE.ordinal());
            addValue("GT",
                    messages.get("manag.pulmonary.sum"),
                    key.toString(),
                    getBlock1Info().getAgeGroupsRows().get(key).getTotal().floatValue())
                    // conditions sent to UI for a detailed case list
                    .addCondition("c.age >= " + key.iniAge)
                    .addCondition("c.age <= " + key.endAge);
        }

        //Total Row
        //Pulmonary - Bacteriologically Confirmed
        for(PatientType pt : getPatientTypesReport()){
            addValue("M"+pt.ordinal()+"bc",
                    messages.get("manag.gender.male"),
                    messages.get("global.total"),
                    getBlock1Info().getTotalByGenderAndPatientType(0, pt, Gender.MALE).floatValue())
                    // conditions sent to UI for a detailed case list
                    .addCondition("p.gender = " + Gender.MALE.ordinal())
                    .addCondition("c.infectionSite = " + InfectionSite.PULMONARY.ordinal())
                    .addCondition("c.caseDefinition = " + CaseDefinition.BACTERIOLOGICALLY_CONFIRMED.ordinal())
                    .addCondition("(c.patientType = " + pt.ordinal() + " or c.previouslyTreatedType = " + pt.ordinal() + ")");

            addValue("F"+pt.ordinal()+"bc",
                    messages.get("manag.gender.female"),
                    messages.get("global.total"),
                    getBlock1Info().getTotalByGenderAndPatientType(0, pt, Gender.FEMALE).floatValue())
                    // conditions sent to UI for a detailed case list
                    .addCondition("p.gender = " + Gender.FEMALE.ordinal())
                    .addCondition("c.infectionSite = " + InfectionSite.PULMONARY.ordinal())
                    .addCondition("c.caseDefinition = " + CaseDefinition.BACTERIOLOGICALLY_CONFIRMED.ordinal())
                    .addCondition("(c.patientType = " + pt.ordinal() + " or c.previouslyTreatedType = " + pt.ordinal() + ")");
        }

        //Pulmonary - Clinically Confirmed
        for(PatientType pt : getPatientTypesReport()){
            addValue("M" + pt.ordinal() + "cc",
                    messages.get("manag.gender.male"),
                    messages.get("global.total"),
                    getBlock1Info().getTotalByGenderAndPatientType(1, pt, Gender.MALE).floatValue())
                    // conditions sent to UI for a detailed case list
                    .addCondition("p.gender = " + Gender.MALE.ordinal())
                    .addCondition("c.infectionSite = " + InfectionSite.PULMONARY.ordinal())
                    .addCondition("c.caseDefinition = " + CaseDefinition.CLINICALLY_DIAGNOSED.ordinal())
                    .addCondition("(c.patientType = " + pt.ordinal() + " or c.previouslyTreatedType = " + pt.ordinal() + ")");

            addValue("F"+pt.ordinal()+"cc",
                    messages.get("manag.gender.female"),
                    messages.get("global.total"),
                    getBlock1Info().getTotalByGenderAndPatientType(1, pt, Gender.FEMALE).floatValue())
                    // conditions sent to UI for a detailed case list
                    .addCondition("p.gender = " + Gender.FEMALE.ordinal())
                    .addCondition("c.infectionSite = " + InfectionSite.PULMONARY.ordinal())
                    .addCondition("c.caseDefinition = " + CaseDefinition.CLINICALLY_DIAGNOSED.ordinal())
                    .addCondition("(c.patientType = " + pt.ordinal() + " or c.previouslyTreatedType = " + pt.ordinal() + ")");

        }

        //Extrapulmonary
        for(PatientType pt : getPatientTypesReport()){
            addValue("M" + pt.ordinal() + "ex",
                    messages.get("manag.gender.male"),
                    messages.get("global.total"),
                    getBlock1Info().getTotalByGenderAndPatientType(2, pt, Gender.MALE).floatValue())
                    // conditions sent to UI for a detailed case list
                    .addCondition("p.gender = " + Gender.MALE.ordinal())
                    .addCondition("c.infectionSite = " + InfectionSite.EXTRAPULMONARY.ordinal())
                    .addCondition("(c.patientType = " + pt.ordinal() + " or c.previouslyTreatedType = " + pt.ordinal() + ")");

            addValue("F"+pt.ordinal()+"ex",
                    messages.get("manag.gender.female"),
                    messages.get("global.total"),
                    getBlock1Info().getTotalByGenderAndPatientType(2, pt, Gender.FEMALE).floatValue())
                    // conditions sent to UI for a detailed case list
                    .addCondition("p.gender = " + Gender.FEMALE.ordinal())
                    .addCondition("c.infectionSite = " + InfectionSite.EXTRAPULMONARY.ordinal())
                    .addCondition("(c.patientType = " + pt.ordinal() + " or c.previouslyTreatedType = " + pt.ordinal() + ")");
        }

        addValue("MT",
                messages.get("manag.gender.male"),
                messages.get("global.total"),
                getBlock1Info().getGrandTotalByGender(Gender.MALE).floatValue())
                .addCondition("p.gender = " + Gender.MALE.ordinal());

        addValue("FT",
                messages.get("manag.gender.female"),
                messages.get("global.total"),
                getBlock1Info().getGrandTotalByGender(Gender.FEMALE).floatValue())
                .addCondition("p.gender = " + Gender.FEMALE.ordinal());

        addValue("GT", messages.get("manag.pulmonary.sum"), messages.get("global.total"), getBlock1Info().getGrantTotal().floatValue());
    }

    public PatientType[] getPatientTypesReport() {
        return TBForm10Block1Query.PATIENT_TYPES;
    }

    /**
     * This class stores the values
     */
    public class TB10Block1Info{
        private LinkedHashMap<Block1AgeRange, TB10Block1RowInfo> ageGroupsRows;

        /**
         * Initializes objects that will store the information of the block 1 report
         */
        public TB10Block1Info(){
            this.ageGroupsRows = new LinkedHashMap<Block1AgeRange, TB10Block1RowInfo>();

            for(Block1AgeRange item : Block1AgeRange.values()){
                this.ageGroupsRows.put(item, new TB10Block1RowInfo());
            }
        }

        /**
         * Method used to set a value to a certain "cell" of the interface table
         * @param ageRange - Identifies the age range of the quantity
         * @param subgroup - Identifies the subgroup of the quantity. 0 for pulmonaryBacteriologicallyConf, 1 for pulmonaryClinicallyConf, 2 for extrapulmonary.
         * @param gender - Identifies the gender of the quantity
         * @param pt - Identifies the patient type of the quantity
         * @param quantity - the quantity
         */
        public void setValue(Block1AgeRange ageRange, int subgroup, Gender gender, PatientType pt, Long quantity){
            TB10Block1RowInfo rowInfo = ageGroupsRows.get(ageRange);

            if(subgroup == 0 && gender.equals(Gender.MALE))
                rowInfo.getPulmonaryBacteriologicallyConf().getMaleValues().put(pt, (rowInfo.getPulmonaryBacteriologicallyConf().getMaleValues().get(pt).longValue()) + quantity.longValue());
            else if(subgroup == 0 && gender.equals(Gender.FEMALE))
                rowInfo.getPulmonaryBacteriologicallyConf().getFemaleValues().put(pt, (rowInfo.getPulmonaryBacteriologicallyConf().getFemaleValues().get(pt).longValue()) + quantity.longValue());
            else if(subgroup == 1 && gender.equals(Gender.MALE))
                rowInfo.getPulmonaryClinicallyConf().getMaleValues().put(pt, (rowInfo.getPulmonaryClinicallyConf().getMaleValues().get(pt).longValue()) + quantity.longValue());
            else if(subgroup == 1 && gender.equals(Gender.FEMALE))
                rowInfo.getPulmonaryClinicallyConf().getFemaleValues().put(pt, (rowInfo.getPulmonaryClinicallyConf().getFemaleValues().get(pt).longValue()) + quantity.longValue());
            else if(subgroup == 2 && gender.equals(Gender.MALE))
                rowInfo.getExtrapulmonary().getMaleValues().put(pt, (rowInfo.getExtrapulmonary().getMaleValues().get(pt).longValue()) + quantity.longValue());
            else if(subgroup == 2 && gender.equals(Gender.FEMALE))
                rowInfo.getExtrapulmonary().getFemaleValues().put(pt, (rowInfo.getExtrapulmonary().getFemaleValues().get(pt).longValue()) + quantity.longValue());
        }

        /**
         * @param subgroup - select the subgroup to totalize. 0 for pulmonaryBacteriologicallyConf, 1 for pulmonaryClinicallyConf, 2 for extrapulmonary.
         * @param patienttype - set the patient type to totalize.
         * @param gender - set the gender to totalize.
         * @return The total for the column selected according to the params.
         */
        public Long getTotalByGenderAndPatientType(int subgroup, PatientType patienttype, Gender gender){
            Long result = new Long(0);
            for(Block1AgeRange key : ageGroupsRows.keySet()){
                if(subgroup == 0 && gender.equals(gender.MALE))
                    result = result.longValue() + ageGroupsRows.get(key).getPulmonaryBacteriologicallyConf().getMaleValues().get(patienttype).longValue();
                else if(subgroup == 0 && gender.equals(gender.FEMALE))
                    result = result.longValue() + ageGroupsRows.get(key).getPulmonaryBacteriologicallyConf().getFemaleValues().get(patienttype).longValue();
                else if(subgroup == 1 && gender.equals(gender.MALE))
                    result = result.longValue() + ageGroupsRows.get(key).getPulmonaryClinicallyConf().getMaleValues().get(patienttype).longValue();
                else if(subgroup == 1 && gender.equals(gender.FEMALE))
                    result = result.longValue() + ageGroupsRows.get(key).getPulmonaryClinicallyConf().getFemaleValues().get(patienttype).longValue();
                else if(subgroup == 2 && gender.equals(gender.MALE))
                    result = result.longValue() + ageGroupsRows.get(key).getExtrapulmonary().getMaleValues().get(patienttype).longValue();
                else if(subgroup == 2 && gender.equals(gender.FEMALE))
                    result = result.longValue() + ageGroupsRows.get(key).getExtrapulmonary().getFemaleValues().get(patienttype).longValue();

            }
            return result;
        }

        /**
         *
         * @param gender identify the gender that will return the grand total
         * @return the very grand total of cases counted by gender
         */
        public Long getGrandTotalByGender(Gender gender){
            Long result = new Long(0);

            for(Block1AgeRange key : getAgeGroupsRows().keySet()){
                result = result.longValue() + ageGroupsRows.get(key).getTotalByGender(gender).longValue();
            }

            return result;
        }

        /**
         *
         * @return the very grant total of the cases counted
         */
        public Long getGrantTotal(){
            return getGrandTotalByGender(Gender.MALE).longValue() + getGrandTotalByGender(Gender.FEMALE).longValue();
        }

        public HashMap<Block1AgeRange, TB10Block1RowInfo> getAgeGroupsRows() {
            return ageGroupsRows;
        }
    }

    public enum Block1AgeRange {

        UNTIL_4("<=4", 0, 4),
        AR_5_14("5 - 14", 5, 14),
        AR_15_24("15 - 24", 15, 24),
        AR_25_34("25 - 34", 25, 34),
        AR_35_44("35 - 44", 35, 44),
        AR_45_54("45 - 54", 45, 54),
        AR_55_64("55 - 64", 55, 64),
        AR_65_MORE(">=65", 65, 1000);

        private String rowLabel;
        private Integer iniAge;
        private Integer endAge;

        Block1AgeRange(String rowLabel, Integer iniAge, Integer endAge) {
            this.rowLabel = rowLabel;
            this.iniAge = iniAge;
            this.endAge = endAge;
        }

        public static Block1AgeRange findAgeRange(int age){
            for (Block1AgeRange ar : values()) {
                if (age >= ar.iniAge && age <= ar.endAge) {
                    return ar;
                }
            }

            throw new RuntimeException("Block1AgeRange: Age range not found");
        }

        @Override
        public String toString() {
            return this.rowLabel;
        }
    }

    public class TB10Block1RowInfo{
        TB10Block1SubGroupInfo pulmonaryBacteriologicallyConf;
        TB10Block1SubGroupInfo pulmonaryClinicallyConf;
        TB10Block1SubGroupInfo extrapulmonary;

        /**
         * Initializes each sub group object that will store information of block 1
         */
        public TB10Block1RowInfo(){
            this.pulmonaryBacteriologicallyConf = new TB10Block1SubGroupInfo();
            this.pulmonaryClinicallyConf = new TB10Block1SubGroupInfo();
            this.extrapulmonary = new TB10Block1SubGroupInfo();
        }

        public Long getTotalByGender(Gender gender){
            return pulmonaryBacteriologicallyConf.getTotalByGender(gender).longValue() + pulmonaryClinicallyConf.getTotalByGender(gender).longValue() + extrapulmonary.getTotalByGender(gender).longValue();
        }

        public Long getTotal(){
            return getTotalByGender(Gender.MALE).longValue() + getTotalByGender(Gender.FEMALE).longValue();
        }

        public TB10Block1SubGroupInfo getPulmonaryBacteriologicallyConf() {
            return pulmonaryBacteriologicallyConf;
        }

        public TB10Block1SubGroupInfo getPulmonaryClinicallyConf() {
            return pulmonaryClinicallyConf;
        }

        public TB10Block1SubGroupInfo getExtrapulmonary() {
            return extrapulmonary;
        }
    }

    public class TB10Block1SubGroupInfo{
        private HashMap<PatientType, Long> maleValues;
        private HashMap<PatientType, Long> femaleValues;

        /**
         * Initializes each patient type column and its sub columns male and female
         */
        public TB10Block1SubGroupInfo(){
            maleValues = new HashMap<PatientType, Long>();
            for(PatientType type : getPatientTypesReport()){
                this.maleValues.put(type, new Long(0));
            }

            femaleValues = new HashMap<PatientType, Long>();
            for(PatientType type : getPatientTypesReport()){
                this.femaleValues.put(type, new Long(0));
            }
        }

        public HashMap<PatientType, Long> getMaleValues() {
            return maleValues;
        }

        public HashMap<PatientType, Long> getFemaleValues() {
            return femaleValues;
        }

        /**
         *
         * @param gender identify the gender asked
         * @return the total by gender for each column of the table.
         */
        public Long getTotalByGender(Gender gender){
            HashMap<PatientType, Long> values;
            Long result = new Long(0);

            if(gender == null)
                return result;
            else if(gender.equals(gender.MALE))
                values = maleValues;
            else
                values = femaleValues;

            for(PatientType key : values.keySet()){
                result = result.longValue() + values.get(key).longValue();
            }
            return result;
        }
    }
}
